import java.awt.*;
import java.util.*;


	
public class Nodes implements Iterable<Nodes.Node>
{
	private Node first;
	public int radius;
	
	public Nodes(Point first, int radius) {
		this.first = new Node(first);
		this.radius = radius;
	}
	
	public void paint(Graphics2D g) {
		for(Node cur : this) {
			final Point p = cur.p;
			g.drawOval(p.x - radius, p.y - radius, radius * 2, radius * 2);
		}
	}
	
	public void paintVeins(Graphics2D g) {
		for(Node cur : this) {
			if( cur.parent != null ) {
				//g.setStroke(new BasicStroke(cur.childBranches + 1));
				g.setStroke(new BasicStroke((float)cur.thickness));
				g.drawLine(cur.parent.p.x, cur.parent.p.y, cur.p.x, cur.p.y);
			}
		}
	}
	
	public void add(Node parent, Point p) {
		parent.addChild(new Node(p));
	}
	
	public boolean overlap(Point p, int radius) {
		for(Node cur : this) {
			final Point check = cur.p;
			final double dx = check.x - p.x;
			final double dy = check.y - p.y;
			final int r = this.radius + radius;
			if( dx*dx + dy*dy <= r*r )
				return true;
		}
		
		return false;
	}
	
	public Point getFirst() {
		return first.p;
	}
	
	public Iterator<Node> iterator() {
		return new NodeIterator(first);
	}
	
	public void calcThickness() {
		//Need Post-order depth-first traversal (iterator is pre-order)
		calcThickness(first);
	}
	
	private void calcThickness(Node n) {
		Node child = n.child;
		while( child != null ) {
			calcThickness(child);
			child = child.nextSibling;
		}
		
		//Now calc self -- need to find thickest TWO children
		if( n.child != null && n.child.nextSibling != null ) {
			Node t1 = null, t2 = null;
			child = n.child;
			while( child != null ) {
				if( t1 == null || child.thickness > t1.thickness ) {
					t2 = t1;
					t1 = child;
				} else if( t2 == null || child.thickness > t2.thickness ) {
					t2 = child;
				}
				child = child.nextSibling;
			}
			
			// Murray's formula: t_par ^ N = t_child1 ^ N + t_child2 ^ N
			final int N = 3;	//Murray's value was 3, MacDonald says this can vary by plant
			n.thickness = Math.pow( Math.pow(t1.thickness, N) + Math.pow(t2.thickness, N), 1.0 / N );
		} else if( n.child != null ) {
			//Has one child, so copy thickness
			n.thickness = n.child.thickness;
		}
	}
	
	public static class Node implements Comparable
	{
		public Point p;
		public Node child = null;
		public Node nextSibling = null;
		public Node parent = null;				//so can propogate thickness
		public int childBranches = 0;			//count this to know thickness of vein (simple method)
		public double thickness = 1.0;
		
		public Node(Point p) {
			this.p = p;
		}
		
		public void addChild(Node n) {
			if( child == null ) {
				child = n;	//one child does not count as a branch
			} else {
				Node lastChild = child;
				while( lastChild.nextSibling != null ) {
					lastChild = lastChild.nextSibling;
				}
				lastChild.nextSibling = n;
				
				childBranches++;
				Node up = parent;
				while( up != null ) {
					up.childBranches++;
					up = up.parent;
				}
			}
			n.parent = this;
		}
		
		/**
		 * Sort first on X, then on Y
		 * This is used so Nodes can be put in a TreeMap
		 */
		public int compareTo(Object o) {
			if( o instanceof Node ) {
				final Node n = (Node)o;
				int dx = n.p.x - p.x;
				if( dx == 0 )
					return n.p.y - p.y;
				return dx;
			}
			return 0;
		}
	}
	
	/**
	 * Implements a pre-order depth-first walk of the tree
	 */
	private class NodeIterator implements Iterator<Node>
	{
		private Stack<Node> pts = new Stack<Node>();
		private Node cur = null;
		
		public NodeIterator(Node first) {
			pts.push(first);
		}
		
		public boolean hasNext() {
			return !pts.empty() || (cur != null && cur.child != null);
		}
		
		public Node next() {
			//Now that the current has not been removed, add it's children to the stack
			if( cur != null ) {
				Node child = cur.child;
				while(child != null) {
					pts.push(child);
					child = child.nextSibling;
				}
			}
			
			//Grab the next point to explore and return it
			cur = pts.empty()? null : pts.pop();
			return cur;
		}
		
		public void remove() {
			if( cur == null )
				return;	//hasn't called next() yet (ever or since calling remove)
			
			if( cur.parent != null ) {
				//Need to remove self from parent's list
				Node check = cur.parent.child;
				if( check == cur ) {
					//First in list, easy to remove self
					cur.parent.child = cur.nextSibling;
				} else {
					//Somewhere in the list, find it
					while( check.nextSibling != cur ) {
						check = check.nextSibling;
					}
					if( check.nextSibling == cur )	//double check here to avoid null ptr, but should never need to
						check.nextSibling = cur.nextSibling;
				}
			} else {
				//Was the first one!
				first = null;
			}
			
			cur = null;	//we removed this one, so have to grab another off the stack
		}
	}
	
}
